//#include <math.h>     ceil, log10f
//#include <strings.h>  strncasecmp
#include <assert.h>

#include "version.h"
#include "rmutil/util.h"

#include "ec.h"
#include "rm_ec.h"

#define INNER_ERROR(x)                                                                             \
    RedisModule_ReplyWithError(ctx, x);                                                            \
    return REDISMODULE_ERR;

RedisModuleType *ECType;

static int GetECKey(RedisModuleCtx *ctx, RedisModuleString *keyName, ECSketch **ec, int mode) {
    // All using this function should call RedisModule_AutoMemory to prevent memory leak
    RedisModuleKey *key = RedisModule_OpenKey(ctx, keyName, mode);
    if (RedisModule_KeyType(key) == REDISMODULE_KEYTYPE_EMPTY) {
        RedisModule_CloseKey(key);
        INNER_ERROR("EC: key does not exist");
    } else if (RedisModule_ModuleTypeGetType(key) != ECType) {
        RedisModule_CloseKey(key);
        INNER_ERROR(REDISMODULE_ERRORMSG_WRONGTYPE);
    }

    *ec = RedisModule_ModuleTypeGetValue(key);
    RedisModule_CloseKey(key);
    return REDISMODULE_OK;
}

static int createEC(RedisModuleCtx *ctx, RedisModuleString **argv, int argc, ECSketch **ec) {
    long long width;
    if (argc == 3) {
        if ((RedisModule_StringToLongLong(argv[2], &width) != REDISMODULE_OK) ||
            width < 1 || width > 100000) {
            INNER_ERROR("EC: invalid width size");
        }
    } else {
        width = 100000;
    }
    *ec = NewECSketch(width, 2);
    return REDISMODULE_OK;
}

static int EC_Create_Cmd(RedisModuleCtx *ctx, RedisModuleString **argv, int argc) {
    if (argc != 2 && argc != 3) {
        return RedisModule_WrongArity(ctx);
    }

    RedisModuleKey *key = RedisModule_OpenKey(ctx, argv[1], REDISMODULE_READ | REDISMODULE_WRITE);
    if (RedisModule_KeyType(key) != REDISMODULE_KEYTYPE_EMPTY) {
        RedisModule_ReplyWithError(ctx, "EC: key already exists");
        goto final;
    }

    ECSketch *ec = NULL;
    if (createEC(ctx, argv, argc, &ec) != REDISMODULE_OK)
        goto final;

    if (RedisModule_ModuleTypeSetValue(key, ECType, ec) == REDISMODULE_ERR) {
        goto final;
    }

    RedisModule_ReplicateVerbatim(ctx);
    RedisModule_ReplyWithSimpleString(ctx, "OK");
final:
    RedisModule_CloseKey(key);
    return REDISMODULE_OK;
}

static int EC_Insert_Cmd(RedisModuleCtx *ctx, RedisModuleString **argv, int argc) {

    if (argc < 3)
        return RedisModule_WrongArity(ctx);

    ECSketch *ec;
    if (GetECKey(ctx, argv[1], &ec, REDISMODULE_READ | REDISMODULE_WRITE) != REDISMODULE_OK) {
        return REDISMODULE_OK;
    }

    int itemCount = argc - 2;
    RedisModule_ReplyWithArray(ctx, itemCount);

    for (int i = 0; i < itemCount; ++i) {
        size_t itemlen;
        const char *item = RedisModule_StringPtrLen(argv[i + 2], &itemlen);
        ECS_IncrBy(ec, item, itemlen, 1);

        RedisModule_ReplyWithNull(ctx);
    }
    RedisModule_ReplicateVerbatim(ctx);
    return REDISMODULE_OK;
}


static int EC_Query_Cmd(RedisModuleCtx *ctx, RedisModuleString **argv, int argc) {
    if (argc < 3)
        return RedisModule_WrongArity(ctx);

    ECSketch *ec;
    if (GetECKey(ctx, argv[1], &ec, REDISMODULE_READ) != REDISMODULE_OK)
        return REDISMODULE_ERR;

    size_t itemlen;
    long long res;
    RedisModule_ReplyWithArray(ctx, argc - 2);
    for (int i = 2; i < argc; ++i) {
        const char *item = RedisModule_StringPtrLen(argv[i], &itemlen);
        res = ECS_Query(ec, item, itemlen);
        RedisModule_ReplyWithLongLong(ctx, res);
    }

    return REDISMODULE_OK;
}

/**************** Module functions *********************************/

static void ECRdbSave(RedisModuleIO *io, void *obj) {
    ECSketch *ec = obj;

    RedisModule_SaveUnsigned(io, ec->width);
    RedisModule_SaveUnsigned(io, ec->depth);
    /*RedisModule_SaveStringBuffer(io, (const char *)ec->HK,
                                 EC_d * (MAX_MEM+10) * BN * sizeof(node));*/
    for (size_t i=0; i<EC_d; i++){
	    for (size_t j=0; j<MAX_MEM; j++){
            for (size_t k=0; k<5; k++){
	    	    RedisModule_SaveUnsigned(io, ec->HK_EC[i][j].C[k]);
	        }
            for (size_t k=0; k<5; k++){
		        RedisModule_SaveUnsigned(io, ec->HK_EC[i][j].fingerprint[k]);
	        }
            RedisModule_SaveUnsigned(io, ec->HK_EC[i][j].flag);
	    }
    }
}

static void *ECRdbLoad(RedisModuleIO *io, int encver) {
    if (encver > EC_ENC_VER) {
        return NULL;
    }
    ECSketch *ec = ECS_CALLOC(1, sizeof(ECSketch));

    ec->width = RedisModule_LoadUnsigned(io);
    ec->depth = RedisModule_LoadUnsigned(io);

    for (size_t i=0; i<EC_d; i++){
	    for (size_t j=0; j<MAX_MEM; j++){
            for (size_t k=0; k<5; k++){
                ec->HK_EC[i][j].C[k] = RedisModule_LoadUnsigned(io);
	        }
            for (size_t k=0; k<5; k++){
                ec->HK_EC[i][j].fingerprint[k] = RedisModule_LoadUnsigned(io);
	        }
            ec->HK_EC[i][j].flag = RedisModule_LoadUnsigned(io);
	    }
    }
   
    return ec;
}

static void ECFree(void *value) { ECS_Destroy(value); }

static size_t ECMemUsage(const void *value) { 
    ECSketch *ec = (ECSketch *)value;
    return sizeof(ec);
}

static int Test_Cmd(RedisModuleCtx *ctx, RedisModuleString **argv, int argc) {
    RedisModule_ReplyWithLongLong(ctx, 123456);
    return REDISMODULE_OK;
}

int ECModule_onLoad(RedisModuleCtx *ctx, RedisModuleString **argv, int argc) {
    // TODO: add option to set defaults from command line and in program
    RedisModule_Log(ctx, "notice", "EC Loaded");
    RedisModuleTypeMethods tm = {.version = REDISMODULE_TYPE_METHOD_VERSION,
                                 .rdb_load = ECRdbLoad,
                                 .rdb_save = ECRdbSave,
                                 .aof_rewrite = RMUtil_DefaultAofRewrite,
                                 .mem_usage = ECMemUsage,
                                 .free = ECFree};

    RedisModule_Log(ctx, "notice", "EC Create Type Start");
    ECType = RedisModule_CreateDataType(ctx, "ECSk-TYPE", EC_ENC_VER, &tm);
    if (ECType == NULL)
        return REDISMODULE_ERR;
    RedisModule_Log(ctx, "notice", "EC Create Type End");

    RedisModule_Log(ctx, "notice", "EC Create CMD Start");
    RMUtil_RegisterWriteDenyOOMCmd(ctx, "ec.create", EC_Create_Cmd);
    RMUtil_RegisterWriteDenyOOMCmd(ctx, "ec.insert", EC_Insert_Cmd);
    RMUtil_RegisterReadCmd(ctx, "ec.query", EC_Query_Cmd);

    RMUtil_RegisterReadCmd(ctx, "test.test", Test_Cmd);
    RedisModule_Log(ctx, "notice", "EC Create CMD End");

    return REDISMODULE_OK;
}
