/* 

The implementation of hot memory of PBSketch

*/

module Hot_Layer (
    input                                                   clk,
    input                                                   rst_n,

    input   [9:0]                                           i_addr_0,
    input   [2:0]                                           i_key_0,
    input   [4:0]                                           i_freq_0,
    input                                                   i_hot_0,

    input   [9:0]                                           i_addr_1,
    input   [2:0]                                           i_key_1,
    input   [4:0]                                           i_freq_1,
    input                                                   i_hot_1,

    input   [1:0]                                           i_time_step,

    output  reg [1:0]                                       o_burst_time,
    output  reg [9:0]                                       o_addr,
    output  reg [2:0]                                       o_key,
    output  reg [4:0]                                       o_freq,
    output  reg                                             burst_valid
);




// hot memory imp
reg         [16:0]                                          rdata_hot_mem_0;
reg         [16:0]                                          rdata_hot_mem_1;
wire        [16:0]                                          rdata_hot_mem_0_w;
wire        [16:0]                                          rdata_hot_mem_1_w;

always @ (posedge clk or negedge rst_n) begin
    if(~rst_n) begin
        rdata_hot_mem_0                         <=          17'b0;
        rdata_hot_mem_1                         <=          17'b0;
    end
    else begin
        rdata_hot_mem_0                         <=          rdata_hot_mem_0_w;
        rdata_hot_mem_1                         <=          rdata_hot_mem_1_w;
    end
end



Hot_Mem    hot_mem_0(
    // wr ports
    .addra                              (i_addr_0_r2),
    .clka                               (clk),
    .dina                               (wdata_hot_mem_0),
    .wea                                (wea_hot_mem_0),
    // rd ports
    .addrb                              (i_addr_0),
    .clkb                               (clk),
    .doutb                              (rdata_hot_mem_0_w)
);


Hot_Mem    hot_mem_1(
    // wr ports
    .addra                              (i_addr_1_r2),
    .clka                               (clk),
    .dina                               (wdata_hot_mem_1),
    .wea                                (wea_hot_mem_1),
    // rd ports
    .addrb                              (i_addr_1),
    .clkb                               (clk),
    .doutb                              (rdata_hot_mem_1_w)
);



// hot memory read
wire    [2:0]                                               key_0_w;
wire    [4:0]                                               pre_freq_0_w;
wire    [4:0]                                               cur_freq_0_w;
wire    [1:0]                                               pre_ts_0_w;
wire    [1:0]                                               cur_ts_0_w;

wire    [2:0]                                               key_1_w;
wire    [4:0]                                               pre_freq_1_w;
wire    [4:0]                                               cur_freq_1_w;
wire    [1:0]                                               pre_ts_1_w;
wire    [1:0]                                               cur_ts_1_w;

assign      key_0_w                                 =       rdata_hot_mem_0[16:14];
assign      pre_freq_0_w                            =       rdata_hot_mem_0[13:9];
assign      cur_freq_0_w                            =       rdata_hot_mem_0[8:4];
assign      pre_ts_0_w                              =       rdata_hot_mem_0[3:2];
assign      cur_ts_0_w                              =       rdata_hot_mem_0[1:0];

assign      key_1_w                                 =       rdata_hot_mem_1[16:14];
assign      pre_freq_1_w                            =       rdata_hot_mem_1[13:9];
assign      cur_freq_1_w                            =       rdata_hot_mem_1[8:4];
assign      pre_ts_1_w                              =       rdata_hot_mem_1[3:2];
assign      cur_ts_1_w                              =       rdata_hot_mem_1[1:0];










// wr hot ram imp
reg         [16:0]                                          wdata_hot_mem_0;
reg         [16:0]                                          wdata_hot_mem_1;
reg                                                         wea_hot_mem_0;
reg                                                         wea_hot_mem_1;


always @ (posedge clk or negedge rst_n) begin
    if(~rst_n) begin
        wdata_hot_mem_0                             <=      17'b0;
        wdata_hot_mem_1                             <=      17'b0;
        wea_hot_mem_0                               <=      1'b0;
        wea_hot_mem_1                               <=      1'b0;
    end
    else begin
        if (i_hot_0_v_r1) begin
            wea_hot_mem_0                           <=      1'b1;
            wea_hot_mem_1                           <=      1'b0;
            wdata_hot_mem_0                         <=      {i_key_0_r1, cur_freq_0_w, i_freq_0_r1, pre_ts_0_w, time_step_r1};
        end
        else if (i_hot_1_v_r1) begin
            wea_hot_mem_1                           <=      1'b1;
            wea_hot_mem_0                           <=      1'b0;
            wdata_hot_mem_1                         <=      {i_key_1_r1, cur_freq_1_w, i_freq_1_r1, pre_ts_1_w, time_step_r1};
        end
        else begin
            wea_hot_mem_0                           <=      1'b0;
            wea_hot_mem_1                           <=      1'b0;
        end
    end
end






// burst detection
always @ (posedge clk or negedge rst_n) begin
    if(~rst_n) begin
        o_burst_time                                <=      'd0;
        o_addr                                      <=      'd0;
        o_key                                       <=      'd0;
        o_freq                                      <=      'd0;
        burst_valid                                 <=      'd0;
    end
    else begin
        if (i_hot_0_v_r1) begin
            if (i_freq_0_r1 / cur_freq_0_w >= 2) begin
                o_burst_time                        <=      time_step_r1;
                o_addr                              <=      i_addr_0_r1;
                o_key                               <=      i_key_0_r1;
                o_freq                              <=      i_freq_0_r1;
                burst_valid                         <=      1'b1;
            end
            else begin
                burst_valid                         <=      1'b0;
            end
        end
        else if (i_hot_1_v_r1) begin
            if (i_freq_1_r1 / cur_freq_1_w >= 2) begin
                o_burst_time                        <=      time_step_r1;
                o_addr                              <=      i_addr_1_r1;
                o_key                               <=      i_key_1_r1;
                o_freq                              <=      i_freq_1_r1;
                burst_valid                         <=      1'b1;
            end
            else begin
                burst_valid                         <=      1'b0;
            end
        end
        else begin
            burst_valid                             <=      1'b0;
        end
    end
end










// caching pipe
reg         [9:0]                                           i_addr_0_r0;
reg         [9:0]                                           i_addr_0_r1;
reg         [9:0]                                           i_addr_0_r2;
reg         [9:0]                                           i_addr_0_r3;
reg         [9:0]                                           i_addr_0_r4;
reg         [9:0]                                           i_addr_0_r5;

reg         [9:0]                                           i_key_0_r0;
reg         [9:0]                                           i_key_0_r1;
reg         [9:0]                                           i_key_0_r2;
reg         [9:0]                                           i_key_0_r3;
reg         [9:0]                                           i_key_0_r4;
reg         [9:0]                                           i_key_0_r5;

reg         [9:0]                                           i_addr_1_r0;
reg         [9:0]                                           i_addr_1_r1;
reg         [9:0]                                           i_addr_1_r2;
reg         [9:0]                                           i_addr_1_r3;
reg         [9:0]                                           i_addr_1_r4;
reg         [9:0]                                           i_addr_1_r5;

reg         [9:0]                                           i_key_1_r0;
reg         [9:0]                                           i_key_1_r1;
reg         [9:0]                                           i_key_1_r2;
reg         [9:0]                                           i_key_1_r3;
reg         [9:0]                                           i_key_1_r4;
reg         [9:0]                                           i_key_1_r5;

reg         [9:0]                                           i_hot_0_v_r0;
reg         [9:0]                                           i_hot_0_v_r1;
reg         [9:0]                                           i_hot_0_v_r2;
reg         [9:0]                                           i_hot_0_v_r3;
reg         [9:0]                                           i_hot_0_v_r4;
reg         [9:0]                                           i_hot_0_v_r5;

reg         [9:0]                                           i_hot_1_v_r0;
reg         [9:0]                                           i_hot_1_v_r1;
reg         [9:0]                                           i_hot_1_v_r2;
reg         [9:0]                                           i_hot_1_v_r3;
reg         [9:0]                                           i_hot_1_v_r4;
reg         [9:0]                                           i_hot_1_v_r5;

reg         [1:0]                                           time_step_r0;
reg         [1:0]                                           time_step_r1;
reg         [1:0]                                           time_step_r2;
reg         [1:0]                                           time_step_r3;
reg         [1:0]                                           time_step_r4;
reg         [1:0]                                           time_step_r5;

reg         [4:0]                                           i_freq_0_r0;
reg         [4:0]                                           i_freq_0_r1;
reg         [4:0]                                           i_freq_0_r2;
reg         [4:0]                                           i_freq_0_r3;
reg         [4:0]                                           i_freq_0_r4;
reg         [4:0]                                           i_freq_0_r5;

reg         [4:0]                                           i_freq_1_r0;
reg         [4:0]                                           i_freq_1_r1;
reg         [4:0]                                           i_freq_1_r2;
reg         [4:0]                                           i_freq_1_r3;
reg         [4:0]                                           i_freq_1_r4;
reg         [4:0]                                           i_freq_1_r5;




always @ (posedge clk or negedge rst_n) begin
    if(~rst_n) begin
        i_addr_0_r0                                 <=      'd0;
        i_addr_0_r1                                 <=      'd0;
        i_addr_0_r2                                 <=      'd0;
        i_addr_0_r3                                 <=      'd0;
        i_addr_0_r4                                 <=      'd0;
        i_addr_0_r5                                 <=      'd0;
        i_addr_1_r0                                 <=      'd0;
        i_addr_1_r1                                 <=      'd0;
        i_addr_1_r2                                 <=      'd0;
        i_addr_1_r3                                 <=      'd0;
        i_addr_1_r4                                 <=      'd0;
        i_addr_1_r5                                 <=      'd0;
        
        i_key_0_r0                                  <=      'd0;
        i_key_0_r1                                  <=      'd0;
        i_key_0_r2                                  <=      'd0;
        i_key_0_r3                                  <=      'd0;
        i_key_0_r4                                  <=      'd0;
        i_key_0_r5                                  <=      'd0;
        i_key_1_r0                                  <=      'd0;
        i_key_1_r1                                  <=      'd0;
        i_key_1_r2                                  <=      'd0;
        i_key_1_r3                                  <=      'd0;
        i_key_1_r4                                  <=      'd0;
        i_key_1_r5                                  <=      'd0;
        
        i_hot_0_v_r0                                <=      'd0;
        i_hot_0_v_r1                                <=      'd0;
        i_hot_0_v_r2                                <=      'd0;
        i_hot_0_v_r3                                <=      'd0;
        i_hot_0_v_r4                                <=      'd0;
        i_hot_0_v_r5                                <=      'd0;
        i_hot_1_v_r0                                <=      'd0;
        i_hot_1_v_r1                                <=      'd0;
        i_hot_1_v_r2                                <=      'd0;
        i_hot_1_v_r3                                <=      'd0;
        i_hot_1_v_r4                                <=      'd0;
        i_hot_1_v_r5                                <=      'd0;
        
        time_step_r0                                <=      'd0;
        time_step_r1                                <=      'd0;
        time_step_r2                                <=      'd0;
        time_step_r3                                <=      'd0;
        time_step_r4                                <=      'd0;
        time_step_r5                                <=      'd0;

        i_freq_0_r0                                 <=      'd0;
        i_freq_0_r1                                 <=      'd0;
        i_freq_0_r2                                 <=      'd0;
        i_freq_0_r3                                 <=      'd0;
        i_freq_0_r4                                 <=      'd0;
        i_freq_0_r5                                 <=      'd0;

        i_freq_1_r0                                 <=      'd0;
        i_freq_1_r1                                 <=      'd0;
        i_freq_1_r2                                 <=      'd0;
        i_freq_1_r3                                 <=      'd0;
        i_freq_1_r4                                 <=      'd0;
        i_freq_1_r5                                 <=      'd0;
    end
    else begin
        i_addr_0_r0                                 <=      i_addr_0;
        i_addr_0_r1                                 <=      i_addr_0_r0;
        i_addr_0_r2                                 <=      i_addr_0_r1;
        i_addr_0_r3                                 <=      i_addr_0_r2;
        i_addr_0_r4                                 <=      i_addr_0_r3;
        i_addr_0_r5                                 <=      i_addr_0_r4;

        i_addr_1_r0                                 <=      i_addr_1;
        i_addr_1_r1                                 <=      i_addr_1_r0;
        i_addr_1_r2                                 <=      i_addr_1_r1;
        i_addr_1_r3                                 <=      i_addr_1_r2;
        i_addr_1_r4                                 <=      i_addr_1_r3;
        i_addr_1_r5                                 <=      i_addr_1_r4;

        i_hot_0_v_r0                                <=      i_hot_0;
        i_hot_0_v_r1                                <=      i_hot_0_v_r0;
        i_hot_0_v_r2                                <=      i_hot_0_v_r1;
        i_hot_0_v_r3                                <=      i_hot_0_v_r2;
        i_hot_0_v_r4                                <=      i_hot_0_v_r3;
        i_hot_0_v_r5                                <=      i_hot_0_v_r4;
        
        i_hot_1_v_r0                                <=      i_hot_1;
        i_hot_1_v_r1                                <=      i_hot_1_v_r0;
        i_hot_1_v_r2                                <=      i_hot_1_v_r1;
        i_hot_1_v_r3                                <=      i_hot_1_v_r2;
        i_hot_1_v_r4                                <=      i_hot_1_v_r3;
        i_hot_1_v_r5                                <=      i_hot_1_v_r4;

        time_step_r0                                <=      i_time_step;
        time_step_r1                                <=      time_step_r0;
        time_step_r2                                <=      time_step_r1;
        time_step_r3                                <=      time_step_r2;
        time_step_r4                                <=      time_step_r3;
        time_step_r5                                <=      time_step_r4;

        i_freq_0_r0                                 <=      i_freq_0;
        i_freq_0_r1                                 <=      i_freq_0_r0;
        i_freq_0_r2                                 <=      i_freq_0_r1;
        i_freq_0_r3                                 <=      i_freq_0_r2;
        i_freq_0_r4                                 <=      i_freq_0_r3;
        i_freq_0_r5                                 <=      i_freq_0_r4;

        i_freq_1_r0                                 <=      i_freq_1;
        i_freq_1_r1                                 <=      i_freq_1_r0;
        i_freq_1_r2                                 <=      i_freq_1_r1;
        i_freq_1_r3                                 <=      i_freq_1_r2;
        i_freq_1_r4                                 <=      i_freq_1_r3;
        i_freq_1_r5                                 <=      i_freq_1_r4;
    end
end



endmodule